﻿#using Tessa.Extensions.Default.Shared.Workflow.Wf;
#using Tessa.Roles;
#using System.Data;
#using Tessa.Extensions.Default.Shared;
#using LinqToDB.Data

// Свойство для получения ID текущего задания.
public Guid CurrentTaskID
{
    get
    {
        return (Guid) (this.SignalObject.Hash.TryGet<IEnumerable<object>>(
            "TaskIDs",
            EmptyHolder<object>.Collection)
            .FirstOrDefault() ?? Guid.Empty);
    }
}

// Метод для установки флага, определяющего необходимость отзыва подпроцессов из данных завершаемого задания
public void FillNeedRevoke(CardTask task)
{
    this.Process.NeedRevoke = task.Card.DynamicEntries.WfResolutions.RevokeChildren;
}

// Метод для заполнения структуры ролей и важных для маршрута полей из данных завершаемого задания
public void FillNextRolesFromTask(CardTask task)
{
    List<object> nextRoles = new List<object>();

    foreach (CardRow row
        in task.Card.Sections.GetOrAddTable("WfResolutionPerformers").Rows.OrderBy(x => x.Get<int>("Order")))
    {
        if (row.State == CardRowState.Deleted)
        {
            continue;
        }

        nextRoles.Add(
            new Dictionary<string, object>
            {
                {
                    "Role",
                    new Dictionary<string, object>
                    {
                        { "ID", row.Get<Guid>("RoleID") },
                        { "Name", row.Get<string>("RoleName") },
                    }
                }
            });
    }

    this.Signal.NextRoles = nextRoles;
    this.Signal.NeedControl = task.Card.DynamicEntries.WfResolutions.WithControl;
    this.Signal.SingleTask = !task.Card.DynamicEntries.WfResolutions.MassCreation;
    this.Signal.MajorPerformer = task.Card.DynamicEntries.WfResolutions.MajorPerformer;
    this.Signal.RoleID = task.Card.DynamicEntries.WfResolutions.ControllerID ?? this.Context.Session.User.ID;
    this.Signal.RoleName = task.Card.DynamicEntries.WfResolutions.ControllerName ?? this.Context.Session.User.Name;
    this.Signal.Planned = task.Card.DynamicEntries.WfResolutions.Planned;
    this.Signal.DurationInDays = task.Card.DynamicEntries.WfResolutions.DurationInDays;
    this.Signal.KindCaption = task.Card.DynamicEntries.WfResolutions.KindCaption;
    this.Signal.AuthorID = task.Card.Sections["WfResolutions"].RawFields["AuthorID"];
    this.Signal.AuthorName = task.Card.DynamicEntries.WfResolutions.AuthorName;

    this.Signal.ParentTaskRowID = this.SignalObject.Hash.TryGet<IEnumerable<object>>(
        "TaskIDs",
        EmptyHolder<object>.Collection)
        .FirstOrDefault();
}

// Метод для заполнения комментария при завершении задания
public void FillCompleteComment(CardTask task)
{
    this.Signal.Comment = task.Card.DynamicEntries.WfResolutions.Comment;
}

// Метод для заполнения структуры ролей и важных для маршрута полей из данных, сохраненных для отправки на контроль 
public void FillNextRolesFromControl()
{
    this.Signal.NextRoles = new List<object>
    {
        new Dictionary<string, object>
        {
            { "Role", this.Context.ProcessInstance.Hash.Get<Dictionary<string, object>>("ControlRole")}
        }
    };

    this.Signal.ParentTaskRowID = this.SignalObject.Hash.TryGet<IEnumerable<object>>("TaskIDs", EmptyHolder<object>.Collection).FirstOrDefault();

    this.Signal.NeedControl = false;
    this.Signal.DurationInDays = 3.0;
    this.Signal.Planned = null;
    this.Signal.MajorPerformer = false;
    this.Signal.SingleTask = false;
    this.Signal.KindCaption = null;
    this.Signal.AuthorID = null;
    this.Signal.AuthorName = null;

    string userName = this.Context.Session.User.Name;
    string comment = (string) this.Signal.Comment;

    this.Signal.Comment = string.IsNullOrEmpty(comment)
                        ? string.Format(LocalizationManager.LocalizeAndEscapeFormat("$WfResolution_Digest_ControlWithoutComment"), userName)
                        : string.Format(LocalizationManager.LocalizeAndEscapeFormat("$WfResolution_Digest_ControlWithComment"), userName, comment);

    this.Process.NeedControl = false;
    this.Process.TasksCount = 1;
}

// Вспомогательный метод для создания агрегатной роли из списка ролей
public async Task<List<object>> AggregateRolesAsync(List<object> roles)
{
    List<Tuple<Guid, string>> performerTuples = new List<Tuple<Guid, string>>();
    foreach (Dictionary<string, object> role in roles)
    {
        performerTuples.Add(
            new Tuple<Guid, string>(
                WorkflowEngineHelper.Get<Guid>(role, "Role", "ID"),
                WorkflowEngineHelper.Get<string>(role, "Role", "Name")));
    }

    if (performerTuples.Count == 1)
    {
        return roles;
    }

    IRoleRepository roleRepository = this.Context.Container.Resolve<IRoleRepository>();
    (bool success, TaskRole performerRole)
        = await WfHelper.TryCreateResolutionPerformerRoleAsync(
               this.Context.ProcessInstance.CardID,
               performerTuples,
               roleRepository,
               this.Context.DbScope,
               this.Context.ValidationResult);

    if (!success)
    {
        return null;
    }

    await roleRepository.InsertAsync(performerRole);

    return new List<object>()
    {
        new Dictionary<string, object>(StringComparer.Ordinal)
        {
            {
                "Role",
                new Dictionary<string, object>(StringComparer.Ordinal)
                {
                    { "ID", performerRole.ID},
                    { "Name", performerRole.Name},
                }
            }
        }
    };
}

// Метод для обновления данных задания в секции WfResolutionChildren
public async Task UpdateChildRecordAsync(
            CardTask task,
            bool isDeleted,
            bool isCompletion)
{
    IQueryBuilderFactory builderFactory = this.Context.DbScope.BuilderFactory;
    IQueryExecutor executor = this.Context.DbScope.Executor;

    // если даже родительское задание уже завершено, то update не упадёт, а просто не обновит строки
    IQueryBuilder builder = builderFactory
        .Update("WfResolutionChildren")
            .C("InProgress").Assign().C("th", "InProgress")
            .C("PerformerID").Assign().C("th", "RoleID")
            .C("PerformerName").Assign().C("th", "RoleName")
            .C("UserID").Assign().C("th", "UserID")
            .C("UserName").Assign().C("th", "UserName");

    if (isDeleted)
    {
        builder
            .C("Completed").Assign().C("th", "Completed")
            .C("OptionID").Assign().C("th", "OptionID")
            .C("OptionCaption").Assign().C("th", "OptionCaption");
    }

    List<DataParameter> parameters = new List<DataParameter>
        {
            executor.Parameter("TaskID", task.RowID),
        };

    if (isCompletion)
    {
        string answer = null;
        Guid? optionID = task.OptionID;
        if (optionID == DefaultCompletionOptions.Complete)
        {
            Card taskCard = task.TryGetCard();
            StringDictionaryStorage<CardSection> taskSections;
            if (taskCard != null
                && (taskSections = taskCard.TryGetSections()) != null
                && taskSections.TryGetValue(WfHelper.ResolutionSection, out CardSection resolutionSection))
            {
                answer = resolutionSection.RawFields.TryGet<string>(WfHelper.ResolutionCommentField);
            }
        }
        else
        {
            // удаляем комментарий автора, если он есть, отделённый переводом строки от прочей информации
            answer = task.Result;
            if (answer != null)
            {
                int firstLineEnding = answer.IndexOf(Environment.NewLine, StringComparison.Ordinal);
                if (firstLineEnding > 0)
                {
                    answer = answer.Substring(firstLineEnding + Environment.NewLine.Length);
                }
            }
        }

        if (answer != null)
        {
            builder
                .C("Answer").Assign().P("Answer");

            parameters.Add(
                executor.Parameter("Answer", answer));
        }
    }

    builder
        .From("TaskHistory", "th").NoLock()
        .Where().C("WfResolutionChildren", "RowID").Equals().C("th", "RowID")
            .And().C("WfResolutionChildren", "RowID").Equals().P("TaskID");

    await executor
        .ExecuteNonQueryAsync(
            builder.Build(),
            CancellationToken.None,
            parameters.ToArray());
}

// Метод для расчета результата задания при отзыве
public void FillRevokeResult(CardTask task)
{
    task.Result = LocalizationManager.GetString("WfResolution_Result_Revoked");
}

// Метод для расчета результата задания при завершении задания с вариантом Завершить
public void FillCompleteTaskResult(CardTask task)
{
    Dictionary<string, object> resolutionFields = task.Card.Sections[WfHelper.ResolutionSection].RawFields;
    string comment = resolutionFields.Get<string>(WfHelper.ResolutionCommentField).NormalizeComment();

    task.Result = string.IsNullOrEmpty(comment)
        ? LocalizationManager.GetString("WfResolution_Result_CompletedWithoutComment")
    : string.Format(LocalizationManager.LocalizeAndEscapeFormat("$WfResolution_Result_Completed"), comment);
}

// Метод для заполнения результата задания при завершении задания с вариантом отправить
public void FillSendToPerformerTaskResult(CardTask task)
{
    string result = null;
    CardRow[] performersRows = WfHelper.TryGetPerformers(task);
    if (performersRows == null)
    {
        return;
    }

    string performers = string.Join(
        "; ",
        performersRows
            .Select(x =>
                LocalizationManager.Localize(
                    x.Get<string>(WfHelper.ResolutionPerformerRoleNameField))));

    Dictionary<string, object> resolutionFields = task.Card.Sections[WfHelper.ResolutionSection].RawFields;
    string comment = resolutionFields.Get<string>(WfHelper.ResolutionCommentField).NormalizeComment();
    bool massCreation = performersRows.Length > 1 && resolutionFields.Get<bool>(WfHelper.ResolutionMassCreationField);

    if (massCreation)
    {
        bool withControl = resolutionFields.Get<bool>(WfHelper.ResolutionWithControlField);
        if (withControl)
        {
            result = string.IsNullOrEmpty(comment)
                ? string.Format(
                    LocalizationManager.LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithMassCreationAndWithControlAndWithoutComment"),
                     performers)
                 : string.Format(
                     LocalizationManager.LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithMassCreationAndWithControlAndWithComment"),
                     performers,
                     comment);
        }
        else
        {
            result = string.IsNullOrEmpty(comment)
                ? string.Format(
                    LocalizationManager.LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithMassCreationAndWithoutComment"),
                    performers)
                : string.Format(
                    LocalizationManager.LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithMassCreationAndWithComment"),
                    performers,
                    comment);
        }
    }
    else
    {
        bool withControl = resolutionFields.Get<bool>(WfHelper.ResolutionWithControlField);
        if (withControl)
        {
            result = string.IsNullOrEmpty(comment)
                ? string.Format(
                    LocalizationManager.LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithControlAndWithoutComment"),
                    performers)
                : string.Format(
                    LocalizationManager.LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithControlAndWithComment"),
                    performers,
                    comment);
        }
        else
        {
            result = string.IsNullOrEmpty(comment)
                ? string.Format(
                    LocalizationManager.LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithoutControlAndWithoutComment"),
                    performers)
                : string.Format(
                    LocalizationManager.LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithoutControlAndWithComment"),
                    performers,
                    comment);
        }
    }
    task.Result = result;
}

// Метод для обновления TypeCaption в строчке истории заданий
public async Task UpdateHistoryCaptionAsync(CardTask task)
{
    string title = this.TryGetTitleFromSection(task);
    if (title != null)
    {
        IQueryExecutor executor = this.Context.DbScope.Executor;
        IQueryBuilderFactory builderFactory = this.Context.DbScope.BuilderFactory;

        await executor
            .ExecuteNonQueryAsync(
                builderFactory
                    .Update("TaskHistory")
                            .C("TypeCaption").Equals().P("Title")
                            .Where().C("RowID").Equals().P("RowID")
                            .Build(),
                CancellationToken.None,
                executor.Parameter("Title", title),
                executor.Parameter("RowID", task.RowID));
    }
}

// Метод для получения заголовка типа задания из секции карточки задания
private string TryGetTitleFromSection(CardTask task)
{
    Card taskCard = task.TryGetCard();
    StringDictionaryStorage<CardSection> taskSections;
    return taskCard != null
           && (taskSections = taskCard.TryGetSections()) != null
           && taskSections.TryGetValue(WfHelper.CommonInfoSection, out CardSection commonInfoSection)
        ? commonInfoSection.RawFields.TryGet<string>(WfHelper.CommonInfoKindCaptionField)
        : null;
}